%% Description
% Plots figure 1a: the adhesion force as a function of H_\infty for
% different fixed \Gamma (powers of 10).
%
% Data has been calculated by solving eqn (14) for non-contact solutions 
% and (16)-(18) for contact solutions for H_\infty from 5-200 and \Gamma in 
% powers of 10 from 1 to 10^4. 

%% Paramter definitions
%
% G(k)      Values of \Gamma used: \Gamma = 10^(k-1).
%
% H(i)      Values of H_\infty used.
%
% j         Denotes the branch of solutions: 
%                       1 = stable non-contact, 
%                       2 = unstable, 
%                       3 = stable contact. 
%
% F(i,j,k)  Dimensionless adhesion force in equilibrium when \Gamma = G(k),
%           H_\infty = H(i) on branch j. F=0 when no such solution exists.
% 
% col       Colour scheme for plotting

clear all;close all;
%% Load calculated data
% Contains values for G, H, F
load('figure2a.mat')

%% Plotting
% Set colour scheme: parula
col = parula(5);

% Plot line for rigid case
figure(2)
x = [1e0 1e2];
loglog(x,2./x.^2,'k--','linewidth',2)
hold on

% Plot force as function of H for \Gamma = 10^{0,1,..4}
for k = 1:5
    
    if k==1
        
        loglog(H,F(:,1,k),'color',col(k,:),'linewidth',2), hold on
        loglog(H,F(:,2,k),'color',col(k,:),'linewidth',2,'linestyle','--')
        loglog(H,F(:,3,k),'color',col(k,:),'linewidth',4)
        
    elseif k==2
        
        % Need to join between contact and not
        % Find where F non-zero first/last for contact/not
        n1 = find(F(:,1,k),1,'first');
        n2 = find(F(:,3,k),1,'last');
        
        loglog([mean([H(n1) H(n2)]) H(n1:end)],...
            [mean([F(n1,1,k) F(n2,3,k)]); F(n1:end,1,k)],...
            'color',col(k,:),'linewidth',2)
        loglog([H(1:n2) mean([H(n1) H(n2)])],...
            [F(1:n2,3,k); mean([F(n1,1,k) F(n2,3,k)])],...
            'color',col(k,:),'linewidth',4)  
        
    else
        
        % Need to join between types of solutions in plot
        % Find where F non-zero first/last for each type of solution
        n1 = find(F(:,1,k),1,'first');
        n2 = find(F(:,2,k),1,'first');
        n3 = find(F(:,2,k),1,'last');
        n4 = find(F(:,3,k),1,'last');
        
        loglog([mean([H(n1) H(n2)]) H(n1:end)],...
            [mean([F(n1,1,k) F(n2,2,k)]); F(n1:end,1,k)],...
            'color',col(k,:),'linewidth',2)
        loglog([mean([H(n1) H(n2)]) H(n2:n3) mean([H(n3) H(n4)])],...
            [mean([F(n1,1,k) F(n2,2,k)]); F(n2:n3,2,k); mean([F(n3,2,k) F(n4,3,k)])],...
            'color',col(k,:),'linewidth',2,'linestyle','--')
        loglog([H(1:n4) mean([H(n3) H(n4)])],...
            [F(1:n4,3,k); mean([F(n3,2,k) F(n4,3,k)])],...
            'color',col(k,:),'linewidth',4)
      
    end
end

% Label axes
xlabel('$H_\infty$','interpreter','Latex')
ylabel('$F$','interpreter','Latex')

% Increase axis label size
set(findobj('type','axes'),'fontsize',20,'linewidth',2)

% Increase tick length
set(gca,'TickLength',2*get(gca,'TickLength'))